<?php

/*
  Plugin Name: Fact Check
  Plugin URI: https://spiralscripts.co.uk
  Description: Allows you to create Fact Check custom post types that generate structured data consistent with Google's ClaimReview data type, for inclusion in Google's search results. See <a href="https://developers.google.com/search/docs/data-types/factcheck">Google Developer Docs</a> for further information
  Author: Fiona Coulter
  Version: 1.0
  Author URI: http://spiralscripts.co.uk/
  License: GPL2+
  
 */
 
if(preg_match('#' . basename(__FILE__) . '#', $_SERVER['PHP_SELF'])) { die('You are not allowed to call this page directly.'); }


define('SPIRALFACTCHECKLINKS', 1);
define('SPIRALFACTCHECKLINKS_PLUGIN_FILE', __FILE__);
define('SPIRALFACTCHECKLINKS_PLUGIN_PATH', plugin_dir_path(__FILE__));
 
require SPIRALFACTCHECKLINKS_PLUGIN_PATH . 'definitions.php';
require SPIRALFACTCHECKLINKS_PLUGIN_PATH . 'helpers'.DIRECTORY_SEPARATOR.'html.php';


if(is_admin())
{
	require SPIRALFACTCHECKLINKS_PLUGIN_PATH . 'fact-check-admin.php';
}


register_activation_hook(__FILE__, 'spiral_fact_check_set_default_options');

function spiral_fact_check_set_default_options()
{
	FactCheckDefinitionsHelper::setDefaultOptions();
	
}



add_action( 'init', 'create_fact_check_post_type' );



function create_fact_check_post_type() {
	register_post_type( 'fact_checks',
		array(
				'labels' => array(
				'name' => 'Fact Checks',
				'singular_name' => 'Fact Check',
				'add_new' => 'Add New',
				'add_new_item' => 'Add New Fact Check',
				'edit' => 'Edit',
				'edit_item' => 'Edit Fact Check',
				'new_item' => 'New Fact Check',
				'view' => 'View',
				'view_item' => 'View Fact Check',
				'search_items' => 'Search Fact Checks',
				'not_found' => 'No Fact Checks found',
				'not_found_in_trash' => 'No Fact Checks found in Trash',
				'parent' => 'Parent Fact Check'
			),
		'public' => true,
		'menu_position' => 20,
		'supports' => array( 'title', 'editor', 'comments', 'thumbnail' ),
		'taxonomies' => array( '' ),
		'menu_icon' => plugins_url( 'fact-check.png', __FILE__ ),
		'has_archive' => false,
		'exclude_from_search' => false 
		)
	);
	
	
	register_taxonomy(
		'fact_checks_claim_type',
		'fact_checks',
		array(
			'labels' => array(
				'name' => 'Claim Type',
				'add_new_item' => 'Add New Claim Type',
				'new_item_name' => "New Claim Type Name"
			),
			'show_ui' => true,
			'meta_box_cb' => false,			
			'show_tagcloud' => false,
			'show_in_quick_edit' => false,
			'hierarchical' => true
		)
	);
	
	
}

// Register function to be called when admin interface is visited
add_action( 'admin_init', 'spiral_fact_check_admin_init' );

// Function to register new meta box for claim review post editor
function spiral_fact_check_admin_init() {
	add_meta_box( 'fact_check_review_details_meta_box', 'Fact Check Details', 'fact_check_display_review_details_meta_box', 'fact_checks', 'normal', 'high' );
	
}


// Function to display meta box contents
function fact_check_display_review_details_meta_box( $fact_check ) { 


	// Retrieve current author and rating based on Fact Check ID
	$claim_author = esc_html( get_post_meta( $fact_check->ID, 'claim_author', true ) );
	$claim_authtype = esc_html( get_post_meta( $fact_check->ID, 'claim_authtype', true ) );
	$claim_rating = intval( get_post_meta( $fact_check->ID, 'claim_rating', true ) );
	$claim_reviewed = esc_html( get_post_meta( $fact_check->ID, 'claim_reviewed', true ) );
	$claim_urls = json_decode( get_post_meta( $fact_check->ID, 'claim_urls', true ), true); 
	
	$rating_system = FactCheckDefinitionsHelper::get_rating_system();
	$author_types = FactCheckDefinitionsHelper::get_author_types();
	
	?>
	<table>
		<tr>
			<td style="width: 150px">Claim Reviewed</td>
			<td><input type='text' size='80' name='fact_check_claim_reviewed' value='<?php echo $claim_reviewed; ?>' /></td>
		</tr>    
		<tr>
			<td style="width: 150px">Claim Author</td>
			<td><input type='text' size='80' name='fact_check_author_name' value='<?php echo $claim_author; ?>' /></td>
		</tr>
		<tr>
			<td style="width: 150px">Author Type</td>
			<td>
				<select style="width: 100px" name="fact_check_author_type">
					<!-- Loop to generate all items in dropdown list -->
					<?php foreach($author_types as $author_type): ?>
                      <option value="<?php echo $author_type; ?>" <?php echo selected( $author_type, $claim_authtype ); ?>><?php echo $author_type; ?></option>
					<?php endforeach; ?>
				</select>
			</td>
		</tr>        
		<tr>
			<td style="width: 150px">Claim Rating</td>
			<td>
				<select style="width: 100px" name="fact_check_rating">
					<!-- Loop to generate all items in dropdown list -->
					<?php foreach($rating_system as $value => $text): ?>
                      <option value="<?php echo $value; ?>" <?php echo selected( $value, $claim_rating ); ?>><?php echo $text; ?></option>
					<?php endforeach; ?>
				</select>
			</td>
		</tr>
		<tr>
			<td>Claim Type</td>
			<td>
				<?php 

				// Retrieve array of types assigned to post
				$assigned_types = wp_get_post_terms( $fact_check->ID, 'fact_checks_claim_type' );
				
				// Retrieve array of all Claim Types in system
				$claim_types = get_terms( 'fact_checks_claim_type', 
                                                         array( 'orderby' => 'name',
                                                                'hide_empty' => 0 ) );
				
				// Check if Claim Types were found
				if ( $claim_types ) {
					echo '<select name="fact_check_claim_type" style="width: 400px">';

					// Display all Claim Types
					foreach ( $claim_types as $claim_type ) {
						echo '<option value="' . $claim_type->term_id . '" ';
						if ( !empty( $assigned_types ) ) {
							selected( $assigned_types[0]->term_id, $claim_type->term_id );
						}
						echo '>' . $claim_type->name . '</option>';
					}
					echo '</select>';
		} ?>
			</td>
		</tr>
        
	</table>
    <fieldset title="Claim URLs" style="margin:10px; padding:10px; border: 1px solid #CCC">
    <legend>Claim URLs</legend>    
    <div class="repeat">
        <table class="wrapper" width="100%">
            <thead>
                <tr>
                    <td width="10%" colspan="4"><span class="add btn">Add + </span></td>
                </tr>
            </thead>
            <tbody class="container">
            <tr class="template row">
                <td width="10%">
                    <span class="move btn">Move</span>
                </td>        
                <td width="10%"><label>Claim URL</label></td>               
                <td width="70%">
                    <input type="text" name="fact_check_claim_urls[{{row-count-placeholder}}]" size="80"/>
                </td>               
                <td width="10%"><span class="remove btn">Remove - </span></td>
            </tr>
            <?php if(count($claim_urls)): ?>
              <?php $i=0; ?>
               <?php foreach($claim_urls as $claim_url): ?>
                  <tr class="row">
                      <td width="10%">
                          <span class="move btn">Move</span>
                      </td>
                      <td width="10%"><label>Claim URL</label></td>                     
                      <td width="70%">
                          <input type="text" name="fact_check_claim_urls[<?php echo $i; ?>]" value="<?php echo esc_html($claim_url); ?>" size="80"/>
                      </td>
                      <td width="10%"><span class="remove btn">Remove - </span></td>
                  </tr>
                  <?php $i++; ?>
               <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>  
    </fieldset> 
    <script type="text/javascript"> 
    jQuery(function() {
        jQuery('.repeat').each(function() {
            jQuery(this).repeatable_fields();
        });
    });
	</script>
<?php }

// Register function to be called when posts are saved
// The function will receive 2 arguments
add_action( 'save_post', 'fact_check_add_fact_check_fields', 10, 2 );

function fact_check_add_fact_check_fields( $post_id = false, $post = false ) {
	// Check post type for Fact Checks
	if ( 'fact_checks' == $post->post_type ) {
		// Store data in post meta table if present in post data
		
		if ( isset( $_POST['fact_check_claim_reviewed'] ) ) {
			update_post_meta( $post_id, 'claim_reviewed', sanitize_text_field( $_POST['fact_check_claim_reviewed'] ) );
		}
		
		
		if ( isset( $_POST['fact_check_author_name'] ) ) {
			update_post_meta( $post_id, 'claim_author', sanitize_text_field( $_POST['fact_check_author_name'] ) );
		}
		
		if ( isset( $_POST['fact_check_author_type'] ) ) {
			update_post_meta( $post_id, 'claim_authtype', sanitize_text_field( $_POST['fact_check_author_type'] ) );
		}		
		
		if ( isset( $_POST['fact_check_rating'] ) && !empty( $_POST['fact_check_rating'] ) ) {
			update_post_meta( $post_id, 'claim_rating', intval( $_POST['fact_check_rating'] ) );
		}
		
		if ( isset( $_POST['fact_check_claim_urls'] ) ) {
			$claim_urls = json_encode($_POST['fact_check_claim_urls']);
			update_post_meta( $post_id, 'claim_urls', sanitize_text_field( $claim_urls ) );
		}		
		
		
		if ( isset( $_POST['fact_check_claim_type'] ) && !empty( $_POST['fact_check_claim_type'] ) ) {
			wp_set_post_terms( $post_id, intval( $_POST['fact_check_claim_type'] ), 'fact_checks_claim_type' );
		}
		
	}
}


add_filter( 'template_include', 'fact_check_template_include', 1 );

function fact_check_template_include( $template_path ){

	
	if ( 'fact_checks' == get_post_type() ) {
		

		if ( is_single() ) {
			// checks if the file exists in the theme first,
			// otherwise install content filter
			if ( $theme_file = locate_template( array( 'single-fact_checks.php' ) ) ) {
				$template_path = $theme_file;
			} else {
				add_filter( 'the_content', 'fact_check_display_single_fact_check', 20 );
			}
		}
		else
		{
			/*if ( $theme_file = locate_template( array( 'archive-fact_checks.php' ) ) ) {
				$template_path = $theme_file;
			} else {
				add_filter( 'the_content', 'fact_check_display_archive_fact_check', 20 );
			}*/
			
			
		}
	}	
	
	return $template_path;
}

function fact_check_display_single_fact_check( $content ) {

    if ( !empty( get_the_ID() ) ) {
			   
	   $content = SpiralFactCheckHTMLHelper::getTemplateOutput('single-fact_checks.php'). $content;
		
	   return $content;
	}
}


add_shortcode( 'fact-check-list', 'fact_check_fact_check_list' );

// Implementation of short code function
function fact_check_fact_check_list() {
	// Preparation of query array to retrieve 5 Fact Checks
	$query_params = array( 'post_type' => 'fact_checks',
                           'post_status' => 'publish',
                           'posts_per_page' => 5 );
	
	// Retrieve page query variable, if present
	$page_num = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

	// If page number is higher than 1, add to query array
	if ( $page_num != 1 ) {
		$query_params['paged'] = $page_num;
	}

	// Execution of post query
	$fact_check_query = new WP_Query;
    $fact_check_query->query( $query_params );
	
	// Check if any posts were returned by query
	if ( $fact_check_query->have_posts() ) {
		// Display posts in table layout
		$output = '<table>';
		$output .= '<tr><th style="width: 350px"><strong>Title</strong></th>';
		$output .= '<th><strong>Author</strong></th></tr>';
		$output .= '<th><strong>Claim</strong></th></tr>';

		// Cycle through all items retrieved
		while ( $fact_check_query->have_posts() ) {
			$fact_check_query->the_post();
			$output .= '<tr><td><a href="' . get_permalink() . '">';
			$output .= get_the_title( get_the_ID() ) . '</a></td>';
			$output .= '<td>' . esc_html( get_post_meta( get_the_ID(), 'claim_author', true ) );
			$output .= '</td><td>' . esc_html( get_post_meta( get_the_ID(), 'claim_reviewed', true ) );
			$output .= '</td></tr>';
		}

		$output .= '</table>';

		// Display page navigation links
		if ( $fact_check_query->max_num_pages > 1 ) {
			$output .= '<nav id="nav-below">';
			$output .= '<div class="nav-previous">';
			$output .= get_next_posts_link( '<span class="meta-nav">&larr;</span> Older reviews', $fact_check_query->max_num_pages );
			$output .= '</div>';
			$output .= "<div class='nav-next'>";
			$output .= get_previous_posts_link( 'Newer reviews <span class="meta-nav">&rarr;</span>', $fact_check_query->max_num_pages );
			$output .= '</div>';
			$output .= '</nav>';
		}

		// Reset post data query
		wp_reset_postdata();
	}

	return $output;
}




add_action( 'fact_checks_claim_type_edit_form_fields', 'fact_check_claim_type_new_fields', 10, 2 );
add_action( 'fact_checks_claim_type_add_form_fields', 'fact_check_claim_type_new_fields', 10, 2 );

function fact_check_claim_type_new_fields( $tag ) {
	$mode = 'new';
	
	if ( is_object( $tag ) ) {
		$mode = 'edit';
		$claim_cat_color = get_term_meta( $tag->term_id, 'claim_type_color', true );
	}
	$claim_cat_color = empty( $claim_cat_color ) ? '#' : $claim_cat_color;

	if ( 'edit' == $mode ) {
		echo '<tr class="form-field">';
		echo '<th scope="row" valign="top">';
	} elseif ( 'new' == $mode ) {
		echo '<div class="form-field">';
	} ?>

	<label for="tag-category-url">Color</label>
	<?php if ( 'edit' == $mode ) {
		echo '</th><td>';
	} ?>

	<input type="text" id="claim_type_color" name="claim_type_color" value="<?php echo $claim_cat_color; ?>" />
	<p class="description">Color associated with Claim Type (e.g. #199C27 or #CCC)</p>

	<?php if ( 'edit' == $mode ) {
		echo '</td></tr>';
	} elseif ( 'new' == $mode ) {
		echo '</div>';
	}
}

add_action( 'edited_fact_checks_claim_type', 'fact_check_save_claim_type_new_fields', 10, 2 );
add_action( 'created_fact_checks_claim_type', 'fact_check_save_claim_type_new_fields', 10, 2 );

function fact_check_save_claim_type_new_fields( $term_id, $tt_id ) {
	if ( !$term_id ) {
		return;
	}

	if ( isset( $_POST['claim_type_color'] ) && ( '#' == $_POST['claim_type_color'] || preg_match( '/#([a-f0-9]{3}){1,2}\b/i', $_POST['claim_type_color'] ) ) ) {
		$returnvalue = update_term_meta( $term_id, 'claim_type_color', sanitize_text_field($_POST['claim_type_color']) ); 
		//note that this sanitization is not really necessary because only colour values are accepted that fit the regular expression
	}
}

// Register function to be called when column list is being prepared
add_filter( 'manage_edit-fact_checks_columns', 'fact_check_add_columns' );

// Function to add columns for author and type in Fact Check listing
// and remove comments columns
function fact_check_add_columns( $columns ) {
	$columns['fact_checks_author'] = 'Author';
	$columns['fact_checks_claim'] = 'Claim';
	$columns['fact_checks_rating'] = 'Rating';
	$columns['fact_checks_type'] = 'Type';
	unset( $columns['comments'] );

	return $columns;
}

// Register function to be called when custom post columns are rendered
add_action( 'manage_posts_custom_column', 'fact_check_populate_columns' );

// Function to send data for custom columns when displaying items
function fact_check_populate_columns( $column ) {
	global $post;
	
	$rating_system = FactCheckDefinitionsHelper::get_rating_system();

	// Check column name and send back appropriate data
	if ( 'fact_checks_author' == $column ) {
		$claim_author = esc_html( get_post_meta( get_the_ID(), 'claim_author', true ) );
		echo $claim_author;
	}
	elseif ( 'fact_checks_claim' == $column ) {
		$claim_reviewed = esc_html( get_post_meta( get_the_ID(), 'claim_reviewed', true ) );
		echo $claim_reviewed;
	}	
	elseif ( 'fact_checks_rating' == $column ) {
		$claim_rating = get_post_meta( get_the_ID(), 'claim_rating', true );
		echo $rating_system[$claim_rating];
	}
	elseif ( 'fact_checks_type' == $column ) {
		$claim_types = wp_get_post_terms( get_the_ID(), 'fact_checks_claim_type' );

		if ( $claim_types ) {
            $claim_cat_color = get_term_meta( $claim_types[0]->term_id, 'claim_type_color', true );

            if ( '#' != $claim_cat_color ) {
                echo '<span style="background-color: ' . esc_html($claim_cat_color);
                echo '; color: #fff; padding: 6px;">';
                echo esc_html($claim_types[0]->name) . '</span>';
            } else {
                echo esc_html($claim_types[0]->name);
            }
        } else {
            echo 'None Assigned'; 
        }
	}
}


add_filter( 'manage_edit-fact_checks_sortable_columns', 'fact_check_author_column_sortable' );

// Register the author and rating columns are sortable columns
function fact_check_author_column_sortable( $columns ) {
	$columns['fact_checks_author'] = 'fact_checks_author';
	$columns['fact_checks_claim'] = 'fact_checks_claim';	
	$columns['fact_checks_rating'] = 'fact_checks_rating';

	return $columns;
}

// Register function to be called when queries are being prepared to
// display post listing
add_filter( 'request', 'fact_check_column_ordering' );

// Function to add elements to query variable based on incoming arguments
function fact_check_column_ordering( $vars ) {
	if ( !is_admin() ) {
		return $vars;
	}
        
	if ( isset( $vars['orderby'] ) && 'fact_checks_author' == $vars['orderby'] ) {
		$vars = array_merge( $vars, array(
				'meta_key' => 'claim_author',
				'orderby' => 'meta_value'
		) );
	}
	elseif ( isset( $vars['orderby'] ) && 'fact_checks_claim' == $vars['orderby'] ) {
		$vars = array_merge( $vars, array(
				'meta_key' => 'claim_reviewed',
				'orderby' => 'meta_value'
		) );
	}	
	elseif ( isset( $vars['orderby'] ) && 'fact_checks_rating' == $vars['orderby'] ) {
		$vars = array_merge( $vars, array(
				'meta_key' => 'claim_rating',
				'orderby' => 'meta_value_num'
		) );
	}

	return $vars;
}

// Register function to be called when displaying post filter drop-down lists
add_action( 'restrict_manage_posts', 'fact_check_claim_type_filter_list' );

// Function to display Claim Type drop-down list for Fact Checks
function fact_check_claim_type_filter_list() {
	$screen = get_current_screen(); 
    global $wp_query; 
	if ( 'fact_checks' == $screen->post_type ) {
		wp_dropdown_categories( array(
			'show_option_all'	=>  'Show All Claim Types',
			'taxonomy'			=>  'fact_checks_claim_type',
			'name'				=>  'fact_checks_claim_type',
			'orderby'			=>  'name',
			'selected'        =>   
            ( isset( $wp_query->query['fact_checks_claim_type'] ) ? 
                 $wp_query->query['fact_checks_claim_type'] : '' ),
			'hierarchical'		=>  false,
			'depth'				=>  3,
			'show_count'		=>  false,
			'hide_empty'		=>  true,
		) );
	}
}

// Register function to be called when preparing post query
add_filter( 'parse_query', 'fact_check_perform_claim_type_filtering' );

// Function to modify query variable based on filter selection
function fact_check_perform_claim_type_filtering( $query ) {
	$qv = &$query->query_vars;

	if ( isset( $qv['fact_checks_claim_type'] ) &&
         !empty( $qv['fact_checks_claim_type'] ) && 
         is_numeric( $qv['fact_checks_claim_type'] ) ) {

			$term = get_term_by( 'id',$qv['fact_checks_claim_type'],'fact_checks_claim_type' );
			$qv['fact_checks_claim_type'] = $term->slug;
    }
}


add_action( 'quick_edit_custom_box', 'fact_check_display_custom_quickedit_link', 10, 2 );

function fact_check_display_custom_quickedit_link( $column_name, $post_type ) {
    if ( 'fact_checks' == $post_type ) {
        switch ( $column_name ) {
            case 'fact_checks_author': ?>
                <?php $author_types = FactCheckDefinitionsHelper::get_author_types(); ?>
                <fieldset class="inline-edit-col-right">
                <div class="inline-edit-col">
                    <label><span class="title">Author</span></label>
                    <input type="text" name='fact_checks_author_input'
                           id='fact_checks_author_input' value="">
                    <label><span class="title">Author Type</span></label>
                    <select name='fact_checks_authtype_input'
                            id='fact_checks_authtype_input'>
                    <?php // Generate all items of drop-down list 
                    foreach($author_types as $author_type) { ?> 
                        <option value="<?php echo $author_type; ?>">
                        <?php echo $author_type; ?> </option> 
                    <?php } ?> 
                    </select>
                </div>
            <?php break;
            case 'fact_checks_claim': ?>
                <fieldset class="inline-edit-col-right">
                <div class="inline-edit-col">
                    <label><span class="title">Claim</span></label>
                    <input type="text" name='fact_checks_claim_input'
                           id='fact_checks_claim_input' value="">
                </div>
            <?php break;			
            case 'fact_checks_rating': ?>
                <?php $rating_system = FactCheckDefinitionsHelper::get_rating_system(); ?>
                <div class="inline-edit-col">
                    <label><span class="title">Rating</span></label>
                    <select name='fact_checks_rating_input'
                            id='fact_checks_rating_input'>
                    <?php // Generate all items of drop-down list 
                    foreach($rating_system as $value => $text) { ?> 
                        <option value="<?php echo $value; ?>">
                        <?php echo $text; ?> </option> 
                    <?php } ?> 
                    </select>
                </div>
            <?php break;
            case 'fact_checks_type': ?>
                <div class="inline-edit-col">
                    <label><span class="title">Type</span></label>
                    <?php
                    $terms = get_terms( 
                             array( 'taxonomy' => 'fact_checks_claim_type',
                                    'hide_empty' => false ) );
                    ?>
                    <select name='fact_checks_type_input'
                            id='fact_checks_type_input'>
                    <?php foreach ($terms as $index => $term) {
                        echo '<option class="fact_checks_type-option"';
                        echo 'value="' . $term->term_id . '"';
                        selected( 0, $index );
                        echo '>' . $term->name. '</option>';
                    } ?>
                    </select>
                </div>
            <?php break;
        } 
    } 
}

add_action( 'admin_footer', 'fact_check_quick_edit_js' );

function fact_check_quick_edit_js() {
    global $current_screen;
    if ( ( 'edit-fact_checks' !== $current_screen->id ) ||
         ( 'fact_checks' !== $current_screen->post_type ) ) {
        return;
    } ?>

    <script type="text/javascript">
    function set_inline_fact_checks( claimReviewArray ) {
        // revert Quick Edit menu so that it refreshes properly
        inlineEditPost.revert();
        var inputClaimAuthor = 
            document.getElementById('fact_checks_author_input');
        inputClaimAuthor.value = claimReviewArray[0];

        var inputClaimAuthtype = 
            document.getElementById('fact_checks_authtype_input');
        for (i = 0; i < inputClaimAuthtype.options.length; i++) {
            if ( inputClaimAuthtype.options[i].value == claimReviewArray[1] ) {
                inputClaimAuthtype.options[i].setAttribute( 'selected',
                                                       'selected' );
            } else {
                inputClaimAuthtype.options[i].removeAttribute( 'selected' );
            }
        } 

        var inputClaimReviewed = 
            document.getElementById('fact_checks_claim_input');
        inputClaimReviewed.value = claimReviewArray[2];
		 
        var inputRating =
            document.getElementById('fact_checks_rating_input');
        for (i = 0; i < inputRating.options.length; i++) {
            if ( inputRating.options[i].value == claimReviewArray[3] ) {
                inputRating.options[i].setAttribute( 'selected',
                                                     'selected' );
            } else {
                inputRating.options[i].removeAttribute( 'selected' );
            }
        } 
 
        var inputClaimType =
            document.getElementById('fact_checks_type_input');
        for (i = 0; i < inputClaimType.options.length; i++) {
            if ( inputClaimType.options[i].value == claimReviewArray[4] ) {
                inputClaimType.options[i].setAttribute( 'selected',
                                                       'selected' );
            } else {
                inputClaimType.options[i].removeAttribute( 'selected' );
            }
        } 
    }
 </script>
 <?php }
 
 add_filter( 'post_row_actions', 'fact_check_quick_edit_link', 10, 2 );
 
 function fact_check_quick_edit_link( $actions, $post ) {
    global $current_screen;
    $post_id = '';

    if ( ( isset( $current_screen ) && 
           $current_screen->id != 'edit-fact_checks' &&
           $current_screen->post_type != 'fact_checks' ) 
         || ( isset( $_POST['screen'] ) &&
              $_POST['screen'] != 'edit-fact_checks' ) ) {
        return $actions;
    }

    if ( !empty( $post->ID ) ) {
        $post_id = $post->ID;
    } elseif ( isset( $_POST['post_ID'] ) ) {
        $post_id = intval( $_POST['post_ID'] );
    }

    if ( !empty( $post_id ) ) {
        $claim_author = esc_html( addslashes(get_post_meta( $post_id, 
                                     'claim_author', true ) )); 
        $claim_authtype = esc_html( get_post_meta( $post_id, 
                                     'claim_authtype', true ) ); 
        $claim_reviewed = esc_html( addslashes(get_post_meta( $post_id, 
                                     'claim_reviewed', true ) )); 
		
        $claim_rating = esc_html( get_post_meta( $post_id, 
                                     'claim_rating', true ) );
        $fact_checks_types = wp_get_post_terms( $post_id, 
                                     'fact_checks_claim_type',
                                     array( 'fields' => 'all' ) );
		if ( empty( $fact_checks_types ) ) {
			$fact_checks_types[0] = (object) array( 'term_id' => 0 );
		}
		
		$fact_checks_types[0]->term_id = (int) $fact_checks_types[0]->term_id;
 
        $idx = 'inline hide-if-no-js';
        $actions[$idx] = '<a href="#" class="editinline" title="';
        $actions[$idx] .= esc_attr( __( 'Edit this item inline' ) ) . '" ';
        $actions[$idx] .= " onclick=\"var claimReviewArray = new Array('";
        $actions[$idx] .= "{$claim_author}', '{$claim_authtype}', '{$claim_reviewed}', '{$claim_rating}', ";
        $actions[$idx] .= "'{$fact_checks_types[0]->term_id}');";
        $actions[$idx] .= "set_inline_fact_checks(claimReviewArray)\">";
        $actions[$idx] .= __( 'Quick&nbsp;Edit' );
        $actions[$idx] .= '</a>';
    }
    return $actions;
}

add_action( 'save_post', 'fact_check_save_quick_edit_data', 10, 2 );

function fact_check_save_quick_edit_data( $ID = false, $post = false ) {
    // Do not save if auto-saving, not Fact Checks, no permissions
    if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) ||
         ( isset( $_POST['post_type'] ) && 'fact_checks' != $_POST['post_type'] ) ||
         !current_user_can( 'edit_page', $ID ) ) {
        return $ID;
    }

    $post = get_post( $ID );
    if ( !empty( $post ) && 'revision' != $post->post_type ) {
        if ( isset( $_POST['fact_checks_author_input'] ) ) {
            update_post_meta( $ID, 'claim_author', 
              sanitize_text_field( $_POST['fact_checks_author_input'] ) ); 
        }
		
        if ( isset( $_POST['fact_checks_authtype_input'] ) ) {
            update_post_meta( $ID, 'claim_authtype', 
              sanitize_text_field( $_POST['fact_checks_authtype_input'] ) ); 
        }
		
		
       if ( isset( $_POST['fact_checks_claim_input'] ) ) {
            update_post_meta( $ID, 'claim_reviewed', 
              sanitize_text_field( $_POST['fact_checks_claim_input'] ) ); 
        }
		
 
        if ( isset( $_POST['fact_checks_rating_input'] ) ) {
            update_post_meta( $ID, 'claim_rating', 
                intval( $_POST['fact_checks_rating_input'] ) ); 
        }
 
        if ( isset( $_POST['fact_checks_type_input'] ) ) {
            $term = term_exists( 
                        intval( $_POST['fact_checks_type_input'] ),
                                'fact_checks_claim_type' );
            if ( !empty( $term ) ) {
                wp_set_object_terms( $ID, 
                    intval( $_POST['fact_checks_type_input'] ), 
                            'fact_checks_claim_type' );
            }
        }
    } 
}


add_filter( 'document_title_parts', 'fact_check_format_fact_check_title' ); 

function fact_check_format_fact_check_title( $the_title ) { 
    if ( 'fact_checks' == get_post_type() && is_single() ) { 
        $claim_author = esc_html( get_post_meta( get_the_ID(),  
                                   'claim_author', true ) ); 
        if ( !empty( $claim_author ) ) {
            $the_title['title'] .= ' by ' . $claim_author;
        }
    } 
 
    return $the_title; 
} 



